// Background

/* jshint esversion:6, -W097 */
/* jslint esversion:6 */
/* global helpers, showError, Worker, chrome, console */

"use strict";


// Chrome don’t follow the W3C specification:
// https://browserext.github.io/browserext/
let bChrome = false;
if (typeof(browser) !== "object") {
    var browser = chrome;
    bChrome = true;
}


const oWorkerHandler = {
    xGCEWorker: null,

    nLastTimeWorkerResponse: 0,  // milliseconds since 1970-01-01

    oTask: {},

    start: function () {
        this.xGCEWorker = new Worker("gce_worker.js");
        this.xGCEWorker.onmessage = function (e) {
            // Messages received from the Worker
            // https://developer.mozilla.org/en-US/docs/Web/API/MessageEvent
            try {
                this.nLastTimeWorkerResponse = Date.now();
                let {sActionDone, result, dInfo, bEnd, bError} = e.data;
                if (bError) {
                    console.log(result);
                    console.log(dInfo);
                    return;
                }
                switch (sActionDone) {
                    case "init":
                        storeGCOptions(result);
                        break;
                    case "parse":
                    case "parseAndSpellcheck":
                    case "parseAndSpellcheck1":
                    case "parseFull":
                    case "getListOfTokens":
                    case "getSpellSuggestions":
                    case "getVerb":
                        // send result to content script
                        if (typeof(dInfo.iReturnPort) === "number") {
                            let xPort = dConnx.get(dInfo.iReturnPort);
                            xPort.postMessage(e.data);
                        } else {
                            console.log("[background] don’t know where to send results");
                            console.log(e.data);
                        }
                        break;
                    case "textToTest":
                    case "fullTests":
                        // send result to panel
                        browser.runtime.sendMessage(e.data);
                        break;
                    case "getOptions":
                    case "getDefaultOptions":
                    case "resetOptions":
                        // send result to panel
                        storeGCOptions(result);
                        browser.runtime.sendMessage(e.data);
                        break;
                    case "setOptions":
                    case "setOption":
                        storeGCOptions(result);
                        break;
                    case "setDictionary":
                    case "setDictionaryOnOff":
                        //console.log("[background] " + sActionDone + ": " + result);
                        break;
                    default:
                        console.log("[background] Unknown command: " + sActionDone);
                        console.log(e.data);
                }
            }
            catch (error) {
                showError(error);
                console.log(e.data);
            }
        };
    },

    getTimeSinceLastResponse: function () {
        // result in seconds
        return Math.floor((Date.now() - this.nLastTimeWorkerResponse) / 1000);
    },

    restart: function (nDelay=5) {
        if (this.getTimeSinceLastResponse() <= nDelay) {
            console.log("Worker not restarted. Worked ", nDelay, " seconds ago.");
            return false;
        }
        if (this.xGCEWorker) {
            this.xGCEWorker.terminate();
        }
        this.start();
        oInitHandler.initGrammarChecker();
        sendCommandToAllTabs("workerRestarted");
        console.log("Worker restarted.");
        return true;
    },

    addTask: function () {
        //
    },

    closeTask: function () {
        //
    }
}


const oInitHandler = {

    initUIOptions: function () {
        if (bChrome) {
            browser.storage.local.get("ui_options", this._initUIOptions);
            browser.storage.local.get("autorefresh_option", this._initUIOptions);
            return;
        }
        browser.storage.local.get("ui_options").then(this._initUIOptions, showError);
        browser.storage.local.get("autorefresh_option").then(this._initUIOptions, showError);
    },

    initGrammarChecker: function () {
        if (bChrome) {
            browser.storage.local.get("gc_options", this._initGrammarChecker);
            browser.storage.local.get("personal_dictionary", this._setSpellingDictionaries);
            browser.storage.local.get("community_dictionary", this._setSpellingDictionaries);
            browser.storage.local.get("oPersonalDictionary", this._setSpellingDictionaries); // deprecated
            browser.storage.local.get("sc_options", this._initSCOptions);
            return;
        }
        browser.storage.local.get("gc_options").then(this._initGrammarChecker, showError);
        browser.storage.local.get("personal_dictionary").then(this._setSpellingDictionaries, showError);
        browser.storage.local.get("community_dictionary").then(this._setSpellingDictionaries, showError);
        browser.storage.local.get("oPersonalDictionary").then(this._setSpellingDictionaries, showError); // deprecated
        browser.storage.local.get("sc_options").then(this._initSCOptions, showError);
    },

    _initUIOptions: function (oSavedOptions) {
        if (!oSavedOptions.hasOwnProperty("ui_options")) {
            browser.storage.local.set({"ui_options": {
                textarea: true,
                editablenode: true
            }});
        }
        if (!oSavedOptions.hasOwnProperty("autorefresh_option")) {
            browser.storage.local.set({"autorefresh_option": true});
        }
    },

    _initGrammarChecker: function (oSavedOptions) {
        try {
            let dOptions = (oSavedOptions.hasOwnProperty("gc_options")) ? oSavedOptions.gc_options : null;
            if (dOptions !== null && Object.getOwnPropertyNames(dOptions).length == 0) {
                console.log("# Error: the saved options was an empty object.");
                dOptions = null;
            }
            oWorkerHandler.xGCEWorker.postMessage({
                sCommand: "init",
                dParam: {sExtensionPath: browser.extension.getURL(""), dOptions: dOptions, sContext: "Firefox"},
                dInfo: {}
            });
        }
        catch (e) {
            console.log("initGrammarChecker failed");
            showError(e);
        }
    },

    _setSpellingDictionaries: function (oData) {
        if (oData.hasOwnProperty("oPersonalDictionary")) {
            // deprecated (to be removed in 2020)
            console.log("personal dictionary migration");
            browser.storage.local.set({ "personal_dictionary": oData["oPersonalDictionary"] });
            oWorkerHandler.xGCEWorker.postMessage({ sCommand: "setDictionary", dParam: { sDictionary: "personal", oDict: oData["oPersonalDictionary"] }, dInfo: {} });
            browser.storage.local.remove("oPersonalDictionary");
        }
        if (oData.hasOwnProperty("community_dictionary")) {
            oWorkerHandler.xGCEWorker.postMessage({ sCommand: "setDictionary", dParam: { sDictionary: "community", oDict: oData["community_dictionary"] }, dInfo: {} });
        }
        if (oData.hasOwnProperty("personal_dictionary")) {
            oWorkerHandler.xGCEWorker.postMessage({ sCommand: "setDictionary", dParam: { sDictionary: "personal", oDict: oData["personal_dictionary"] }, dInfo: {} });
        }
    },

    _initSCOptions: function (oData) {
        if (!oData.hasOwnProperty("sc_options")) {
            browser.storage.local.set({"sc_options": {
                community: true,
                personal: true
            }});
            oWorkerHandler.xGCEWorker.postMessage({ sCommand: "setDictionaryOnOff", dParam: { sDictionary: "community", bActivate: true }, dInfo: {} });
            oWorkerHandler.xGCEWorker.postMessage({ sCommand: "setDictionaryOnOff", dParam: { sDictionary: "personal", bActivate: true }, dInfo: {} });
        } else {
            oWorkerHandler.xGCEWorker.postMessage({ sCommand: "setDictionaryOnOff", dParam: { sDictionary: "community", bActivate: oData.sc_options["community"] }, dInfo: {} });
            oWorkerHandler.xGCEWorker.postMessage({ sCommand: "setDictionaryOnOff", dParam: { sDictionary: "personal", bActivate: oData.sc_options["personal"] }, dInfo: {} });
        }
    }
}

// start the Worker for the GC
oWorkerHandler.start();

// init the options stuff and start the GC
oInitHandler.initUIOptions();
oInitHandler.initGrammarChecker();


// When the extension is installed or updated
browser.runtime.onInstalled.addListener(function (oDetails) {
    // launched at installation or update
    // https://developer.mozilla.org/fr/Add-ons/WebExtensions/API/runtime/onInstalled
    if (oDetails.reason == "update"  ||  oDetails.reason == "installed") {
        // todo
        //browser.tabs.create({url: "http://grammalecte.net"});
    }
});



/*
    Ports from content-scripts
*/

let dConnx = new Map();


/*
    Messages from the extension (not the Worker)
*/
function handleMessage (oRequest, xSender, sendResponse) {
    // message from panels
    //console.log(xSender);
    let {sCommand, dParam, dInfo} = oRequest;
    switch (sCommand) {
        case "getOptions":
        case "getDefaultOptions":
        case "setOptions":
        case "setOption":
        case "resetOptions":
        case "textToTest":
        case "fullTests":
        case "setDictionary":
        case "setDictionaryOnOff":
            oWorkerHandler.xGCEWorker.postMessage(oRequest);
            break;
        case "restartWorker":
            oWorkerHandler.restart(dParam["nDelayLimit"]);
            break;
        case "openURL":
            browser.tabs.create({url: dParam.sURL});
            break;
        case "openConjugueurTab":
            openConjugueurTab();
            break;
        case "openLexiconEditor":
            openLexiconEditor(dParam["dictionary"]);
            break;
        case "openDictionaries":
            openDictionaries();
            break;
        default:
            console.log("[background] Unknown command: " + sCommand);
            console.log(oRequest);
    }
    //sendResponse({response: "response from background script"});
}

browser.runtime.onMessage.addListener(handleMessage);


function handleConnexion (xPort) {
    // Messages from tabs
    let iPortId = xPort.sender.tab.id; // identifier for the port: each port can be found at dConnx[iPortId]
    dConnx.set(iPortId, xPort);
    xPort.onMessage.addListener(function (oRequest) {
        let {sCommand, dParam, dInfo} = oRequest;
        switch (sCommand) {
            case "parse":
            case "parseAndSpellcheck":
            case "parseAndSpellcheck1":
            case "parseFull":
            case "getListOfTokens":
            case "getSpellSuggestions":
            case "getVerb":
                oRequest.dInfo.iReturnPort = iPortId; // we pass the id of the return port to receive answer
                oWorkerHandler.xGCEWorker.postMessage(oRequest);
                break;
            case "restartWorker":
                oWorkerHandler.restart(dParam["nDelayLimit"]);
                break;
            case "openURL":
                browser.tabs.create({url: dParam.sURL});
                break;
            case "openConjugueurTab":
                openConjugueurTab();
                break;
            case "openConjugueurWindow":
                openConjugueurWindow();
                break;
            case "openLexiconEditor":
                openLexiconEditor();
                break;
            default:
                console.log("[background] Unknown command: " + sCommand);
                console.log(oRequest);
        }
    });
    //xPort.postMessage({sActionDone: "newId", result: iPortId});
    xPort.postMessage({sActionDone: "init", sUrl: browser.extension.getURL("")});
}

browser.runtime.onConnect.addListener(handleConnexion);


/*
    Context Menu
*/
// Analyze
browser.contextMenus.create({ id: "grammar_checker_editable",   title: "Analyser cette zone de texte",              contexts: ["editable"] });
browser.contextMenus.create({ id: "grammar_checker_selection",   title: "Analyser la sélection",                     contexts: ["selection"] });
browser.contextMenus.create({ id: "grammar_checker_page",       title: "Analyser la page",                          contexts: ["all"] });
browser.contextMenus.create({ id: "separator_tools",            type: "separator",                                  contexts: ["all"] });
// Tools
browser.contextMenus.create({ id: "conjugueur_tab",             title: "Conjugueur [onglet]",                       contexts: ["all"] });
browser.contextMenus.create({ id: "conjugueur_window",          title: "Conjugueur [fenêtre]",                      contexts: ["all"] });
//browser.contextMenus.create({ id: "dictionaries",               title: "Dictionnaires",                             contexts: ["all"] });
browser.contextMenus.create({ id: "lexicon_editor",             title: "Éditeur lexical",                           contexts: ["all"] });
// Rescan page
browser.contextMenus.create({ id: "separator_rescan",           type: "separator",                                  contexts: ["editable"] });
browser.contextMenus.create({ id: "rescanPage",                 title: "Rechercher à nouveau les zones de texte",   contexts: ["editable"] });


browser.contextMenus.onClicked.addListener(function (xInfo, xTab) {
    // xInfo = https://developer.mozilla.org/en-US/Add-ons/WebExtensions/API/contextMenus/OnClickData
    // xTab = https://developer.mozilla.org/en-US/Add-ons/WebExtensions/API/tabs/Tab
    // confusing: no way to get the node where we click?!
    switch (xInfo.menuItemId) {
        // analyze
        case "grammar_checker_editable":
        case "grammar_checker_page":
            sendCommandToTab(xTab.id, xInfo.menuItemId);
            break;
        case "grammar_checker_selection":
            sendCommandToTab(xTab.id, xInfo.menuItemId, xInfo.selectionText);
            oWorkerHandler.xGCEWorker.postMessage({
                sCommand: "parseAndSpellcheck",
                dParam: {sText: xInfo.selectionText, sCountry: "FR", bDebug: false, bContext: false},
                dInfo: {iReturnPort: xTab.id}
            });
            break;
        // tools
        case "conjugueur_window":
            openConjugueurWindow();
            break;
        case "conjugueur_tab":
            openConjugueurTab();
            break;
        case "lexicon_editor":
            openLexiconEditor();
            break;
        case "dictionaries":
            openDictionaries();
            break;
        // rescan page
        case "rescanPage":
            let xPort = dConnx.get(xTab.id);
            xPort.postMessage({sActionDone: "rescanPage"});
            break;
        default:
            console.log("[Background] Unknown menu id: " + xInfo.menuItemId);
            console.log(xInfo);
            console.log(xTab);
    }
});


/*
    Keyboard shortcuts
*/
browser.commands.onCommand.addListener(function (sCommand) {
    switch (sCommand) {
        case "grammar_checker":
            sendCommandToCurrentTab("shortcutGrammarChecker");
            break;
        case "conjugueur_tab":
            openConjugueurTab();
            break;
        case "lexicon_editor":
            openLexiconEditor();
            break;
        case "dictionaries":
            openDictionaries();
            break;
    }
});


/*
    Tabs
*/
let nTabLexiconEditor = null;
let nTabDictionaries = null;
let nTabConjugueur = null;

browser.tabs.onRemoved.addListener(function (nTabId, xRemoveInfo) {
    switch (nTabId) {
        case nTabLexiconEditor: nTabLexiconEditor = null; break;
        case nTabDictionaries:  nTabDictionaries = null; break;
        case nTabConjugueur:    nTabConjugueur = null; break;
    }
});


/*
    Actions
*/

function storeGCOptions (dOptions) {
    if (dOptions instanceof Map) {
        dOptions = helpers.mapToObject(dOptions);
    }
    browser.storage.local.set({"gc_options": dOptions});
}

function sendCommandToTab (iTab, sCommand, result=null) {
    let xTabPort = dConnx.get(iTab);
    xTabPort.postMessage({sActionDone: sCommand, result: result, dInfo: null, bEnd: false, bError: false});
}

function sendCommandToCurrentTab (sCommand) {
    if (bChrome) {
        browser.tabs.query({ currentWindow: true, active: true }, (lTabs) => {
            for (let xTab of lTabs) {
                console.log(xTab);
                browser.tabs.sendMessage(xTab.id, {sActionRequest: sCommand});
            }
        });
        return;
    }
    browser.tabs.query({ currentWindow: true, active: true }).then((lTabs) => {
        for (let xTab of lTabs) {
            console.log(xTab);
            browser.tabs.sendMessage(xTab.id, {sActionRequest: sCommand});
        }
    }, showError);
}

function sendCommandToAllTabs (sCommand) {
    for (let [iTab, xTabPort] of dConnx.entries()) {
        xTabPort.postMessage({sActionDone: sCommand, result: null, dInfo: null, bEnd: false, bError: false});
    }
}

function openLexiconEditor (sName="__personal__") {
    if (nTabLexiconEditor === null) {
        if (bChrome) {
            browser.tabs.create({
                url: browser.extension.getURL("panel/lex_editor.html")
            }, onLexiconEditorOpened);
            return;
        }
        let xLexEditor = browser.tabs.create({
            url: browser.extension.getURL("panel/lex_editor.html")
        });
        xLexEditor.then(onLexiconEditorOpened, showError);
    }
    else {
        browser.tabs.update(nTabLexiconEditor, {active: true});
    }
}

function onLexiconEditorOpened (xTab) {
    nTabLexiconEditor = xTab.id;
}

function openDictionaries () {
    if (nTabDictionaries === null) {
        if (bChrome) {
            browser.tabs.create({
                url: browser.extension.getURL("panel/dictionaries.html")
            }, onDictionariesOpened);
            return;
        }
        let xLexEditor = browser.tabs.create({
            url: browser.extension.getURL("panel/dictionaries.html")
        });
        xLexEditor.then(onDictionariesOpened, showError);
    }
    else {
        browser.tabs.update(nTabDictionaries, {active: true});
    }
}

function onDictionariesOpened (xTab) {
    nTabDictionaries = xTab.id;
}

function openConjugueurTab () {
    if (nTabConjugueur === null) {
        if (bChrome) {
            browser.tabs.create({
                url: browser.extension.getURL("panel/conjugueur.html")
            }, onConjugueurOpened);
            return;
        }
        let xConjTab = browser.tabs.create({
            url: browser.extension.getURL("panel/conjugueur.html")
        });
        xConjTab.then(onConjugueurOpened, showError);
    }
    else {
        browser.tabs.update(nTabConjugueur, {active: true});
    }
}

function onConjugueurOpened (xTab) {
    nTabConjugueur = xTab.id;
}

function openConjugueurWindow () {
    if (bChrome) {
        browser.windows.create({
            url: browser.extension.getURL("panel/conjugueur.html"),
            type: "popup",
            width: 710,
            height: 980
        });
        return;
    }
    let xConjWindow = browser.windows.create({
        url: browser.extension.getURL("panel/conjugueur.html"),
        type: "popup",
        width: 710,
        height: 980
    });
}


function showError (e) {
    console.error(e);
    //console.error(e.fileName + "\n" + e.name + "\nline: " + e.lineNumber + "\n" + e.message);
}
